<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Manual_recurring_invoices extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('invoices_model');
    }

    /**
     * List all recurring invoices
     */
    public function index()
    {
        if (staff_cant('view', 'invoices')) {
            access_denied('invoices');
        }

        $data['title'] = _l('manual_recurring_invoices');
        $this->load->view('manage', $data);
    }

    /**
     * Get recurring invoices for datatable
     */
    public function table()
    {
        if (staff_cant('view', 'invoices')) {
            ajax_access_denied();
        }

        $this->app->get_table_data(module_views_path('manual_recurring_invoices', 'table'));
    }

    /**
     * Generate the next invoice from a recurring invoice
     */
    public function generate_invoice($id)
    {
        if (staff_cant('create', 'invoices')) {
            access_denied('invoices');
        }

        if (!$id) {
            set_alert('danger', _l('manual_recurring_invoices_invalid_invoice'));
            redirect(admin_url('manual_recurring_invoices'));
        }

        // Get the recurring invoice
        $invoice = $this->invoices_model->get($id);

        if (!$invoice) {
            set_alert('danger', _l('invoice_not_found'));
            redirect(admin_url('manual_recurring_invoices'));
        }

        // Check if this is actually a recurring invoice
        if ($invoice->recurring == 0) {
            set_alert('warning', _l('manual_recurring_invoices_not_recurring'));
            redirect(admin_url('manual_recurring_invoices'));
        }

        // Check if cycles are completed
        if ($invoice->cycles > 0 && $invoice->cycles == $invoice->total_cycles) {
            set_alert('warning', _l('manual_recurring_invoices_cycles_completed'));
            redirect(admin_url('manual_recurring_invoices'));
        }

        // Generate the new invoice
        $new_invoice_id = $this->generate_recurring_invoice($invoice);

        if ($new_invoice_id) {
            set_alert('success', _l('manual_recurring_invoices_generated_successfully', format_invoice_number($new_invoice_id)));
            redirect(admin_url('invoices/list_invoices/' . $new_invoice_id));
        } else {
            set_alert('danger', _l('manual_recurring_invoices_generation_failed'));
            redirect(admin_url('manual_recurring_invoices'));
        }
    }

    /**
     * Generate a new invoice from recurring invoice
     * Based on the logic from Cron_model::recurring_invoices()
     */
    private function generate_recurring_invoice($invoice)
    {
        // Determine the creation date for the new invoice
        if (!$invoice->last_recurring_date) {
            $last_recurring_date = date('Y-m-d', strtotime($invoice->date));
        } else {
            $last_recurring_date = date('Y-m-d', strtotime($invoice->last_recurring_date));
        }

        if ($invoice->custom_recurring == 0) {
            $recurring_type = 'MONTH';
        } else {
            $recurring_type = $invoice->recurring_type;
        }

        $re_create_at = date('Y-m-d', strtotime('+' . $invoice->recurring . ' ' . strtoupper($recurring_type), strtotime($last_recurring_date)));

        // Get the new recurring invoice action setting
        $new_recurring_invoice_action = get_option('new_recurring_invoice_action');

        // Prepare new invoice data
        $new_invoice_data             = [];
        $new_invoice_data['clientid'] = $invoice->clientid;
        $new_invoice_data['number']   = get_option('next_invoice_number');
        $new_invoice_data['date']     = _d($re_create_at);
        $new_invoice_data['duedate']  = null;

        if ($invoice->duedate) {
            $dStart                      = new DateTime($invoice->date);
            $dEnd                        = new DateTime($invoice->duedate);
            $dDiff                       = $dStart->diff($dEnd);
            $new_invoice_data['duedate'] = _d(date('Y-m-d', strtotime('+' . $dDiff->days . ' DAY', strtotime($re_create_at))));
        } else {
            if (get_option('invoice_due_after') != 0) {
                $new_invoice_data['duedate'] = _d(date('Y-m-d', strtotime('+' . get_option('invoice_due_after') . ' DAY', strtotime($re_create_at))));
            }
        }

        $new_invoice_data['project_id']       = $invoice->project_id;
        $new_invoice_data['show_quantity_as'] = $invoice->show_quantity_as;
        $new_invoice_data['currency']         = $invoice->currency;
        $new_invoice_data['subtotal']         = $invoice->subtotal;
        $new_invoice_data['total']            = $invoice->total;
        $new_invoice_data['adjustment']       = $invoice->adjustment;
        $new_invoice_data['discount_percent'] = $invoice->discount_percent;
        $new_invoice_data['discount_total']   = $invoice->discount_total;
        $new_invoice_data['discount_type']    = $invoice->discount_type;
        $new_invoice_data['terms']            = clear_textarea_breaks($invoice->terms);
        $new_invoice_data['sale_agent']       = $invoice->sale_agent;
        $new_invoice_data['billing_street']   = clear_textarea_breaks($invoice->billing_street);
        $new_invoice_data['billing_city']     = $invoice->billing_city;
        $new_invoice_data['billing_state']    = $invoice->billing_state;
        $new_invoice_data['billing_zip']      = $invoice->billing_zip;
        $new_invoice_data['billing_country']  = $invoice->billing_country;
        $new_invoice_data['shipping_street']  = clear_textarea_breaks($invoice->shipping_street);
        $new_invoice_data['shipping_city']    = $invoice->shipping_city;
        $new_invoice_data['shipping_state']   = $invoice->shipping_state;
        $new_invoice_data['shipping_zip']     = $invoice->shipping_zip;
        $new_invoice_data['shipping_country'] = $invoice->shipping_country;
        
        if ($invoice->include_shipping == 1) {
            $new_invoice_data['include_shipping'] = $invoice->include_shipping;
        }
        
        $new_invoice_data['show_shipping_on_invoice'] = $invoice->show_shipping_on_invoice;
        
        // Determine status based on settings
        if ($new_recurring_invoice_action == 'generate_and_send' || $new_recurring_invoice_action == 'generate_unpaid') {
            $new_invoice_data['status'] = 1;
        } elseif ($new_recurring_invoice_action == 'generate_draft') {
            $new_invoice_data['save_as_draft'] = true;
        }
        
        $new_invoice_data['clientnote']            = clear_textarea_breaks($invoice->clientnote);
        $new_invoice_data['adminnote']             = '';
        $new_invoice_data['allowed_payment_modes'] = unserialize($invoice->allowed_payment_modes);
        $new_invoice_data['is_recurring_from']     = $invoice->id;
        $new_invoice_data['newitems']              = [];

        $key                     = 1;
        $custom_fields_items     = get_custom_fields('items');
        
        foreach ($invoice->items as $item) {
            $new_invoice_data['newitems'][$key]['description']      = $item['description'];
            $new_invoice_data['newitems'][$key]['long_description'] = clear_textarea_breaks($item['long_description']);
            $new_invoice_data['newitems'][$key]['qty']              = $item['qty'];
            $new_invoice_data['newitems'][$key]['unit']             = $item['unit'];
            $new_invoice_data['newitems'][$key]['taxname']          = [];
            
            $taxes = get_invoice_item_taxes($item['id']);
            foreach ($taxes as $tax) {
                array_push($new_invoice_data['newitems'][$key]['taxname'], $tax['taxname']);
            }
            
            $new_invoice_data['newitems'][$key]['rate']  = $item['rate'];
            $new_invoice_data['newitems'][$key]['order'] = $item['item_order'];

            foreach ($custom_fields_items as $cf) {
                $new_invoice_data['newitems'][$key]['custom_fields']['items'][$cf['id']] = get_custom_field_value($item['id'], $cf['id'], 'items', false);

                if (!defined('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST')) {
                    define('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST', true);
                }
            }
            $key++;
        }

        $id = $this->invoices_model->add($new_invoice_data);
        
        if ($id) {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'invoices', [
                'addedfrom'                => $invoice->addedfrom,
                'sale_agent'               => $invoice->sale_agent,
                'cancel_overdue_reminders' => $invoice->cancel_overdue_reminders,
            ]);

            $tags = get_tags_in($invoice->id, 'invoice');
            handle_tags_save($tags, $id, 'invoice');

            // Copy custom fields
            $custom_fields = get_custom_fields('invoice');
            foreach ($custom_fields as $field) {
                $value = get_custom_field_value($invoice->id, $field['id'], 'invoice', false);
                if ($value != '') {
                    $this->db->insert(db_prefix() . 'customfieldsvalues', [
                        'relid'   => $id,
                        'fieldid' => $field['id'],
                        'fieldto' => 'invoice',
                        'value'   => $value,
                    ]);
                }
            }

            // Update last recurring date
            $this->db->where('id', $invoice->id);
            $this->db->update(db_prefix() . 'invoices', [
                'last_recurring_date' => $re_create_at,
            ]);

            $this->db->where('id', $invoice->id);
            $this->db->set('total_cycles', 'total_cycles+1', false);
            $this->db->update(db_prefix() . 'invoices');

            if ($new_recurring_invoice_action == 'generate_and_send') {
                $this->invoices_model->send_invoice_to_client($id, 'invoice_send_to_customer', true);
            }

            hooks()->do_action('manual_recurring_invoice_generated', ['original_invoice' => $invoice, 'new_invoice_id' => $id]);
            
            log_activity('Manual Recurring Invoice Generated [Invoice ID: ' . $id . ' from Recurring Invoice: ' . $invoice->id . ']');

            return $id;
        }

        return false;
    }
}